function eng = scaleEngineMap(eng_name, rescaled_eng_name, rescaled_eng_power)

if strcmp(eng_name, rescaled_eng_name)
    error("")
end

eng = loadEngineMap(eng_name);

% Scaling factor
scalingFactor = rescaled_eng_power/eng.ratedPwr;

% main data
eng.displacement = eng.displacement * scalingFactor;
eng.ratedPwr = rescaled_eng_power;
eng.inertia = eng.inertia * scalingFactor;
eng.mass = eng.mass * scalingFactor;

% Torque limit map
eng.maxTrq.Values = eng.maxTrq.Values * scalingFactor;
eng.motTrq.Values = eng.motTrq.Values * scalingFactor;

% Map data
eng.mapData{:, 2:end} = eng.mapData{:, 2:end} * scalingFactor;
mapData = eng.mapData;

% Interpolate data over a new defined grid
% %%% DEFINE THE INTERPOLATION GRID HERE %%%
% Create new uniform grid
spdBrk = eng.idleSpd:100:eng.maxSpd;
trqBrk = min(eng.motTrq.Values):50:(max(eng.maxTrq.Values)+50);

% Create fuel consumption map
% force fc = 0 at the motoring curve
% emptycell = repelem({zeros(size(eng.motTrq.GridVectors{1}))}, 1, width(mapData)-2);
% motCurveData = table(eng.motTrq.GridVectors{1}, eng.motTrq.Values, emptycell{:}, 'VariableNames', mapData.Properties.VariableNames);
% mapData = [mapData; motCurveData];

map_method = 'scattered_interpolant';
% map_method = 'gridfit';
switch map_method
    case 'scattered_interpolant'
        % Note: by default, linear extrapolation is allowed.
        % Uncomment the second line to disable extrapolation
        fc_interpolant = scatteredInterpolant(mapData.speed, mapData.torque, mapData.fuelConsumption);
        nox_interpolant = scatteredInterpolant(mapData.speed, mapData.torque, mapData.NOx);
        hc_interpolant = scatteredInterpolant(mapData.speed, mapData.torque, mapData.HC);
        
        [spdMesh, trqMesh] = ndgrid(spdBrk, trqBrk);
        fc = fc_interpolant(spdMesh, trqMesh);
        nox = nox_interpolant(spdMesh, trqMesh);
        hc = hc_interpolant(spdMesh, trqMesh);
    case 'gridfit'
        [fc, ~, ~] = gridfit(mapData.speed, mapData.torque, mapData.fuelConsumption, spdBrk, trqBrk);
        fc = fc';
        [nox, ~, ~] = gridfit(mapData.speed, mapData.torque, mapData.NOx, spdBrk, trqBrk);
        nox = nox';
        [hc, ~, ~] = gridfit(mapData.speed, mapData.torque, mapData.HC, spdBrk, trqBrk);
        hc = hc';
end

% Store FC map  FC = f(Spd, Trq), g/s = FC(rad/s, Nm)
eng.fuelConsumption = griddedInterpolant(spdMesh.*pi/30, trqMesh, fc);
% Store NOx map  NOx = f(Spd, Trq), g/h = NOx(rad/s, Nm)
eng.NOx = griddedInterpolant(spdMesh.*pi/30, trqMesh, nox);
% Store HC map  HC = f(Spd, Trq), g/h = HC(rad/s, Nm)
eng.HC = griddedInterpolant(spdMesh.*pi/30, trqMesh, hc);

% Save to spreadsheet
saveEngineMap(eng, rescaled_eng_name)

end

